
/* fm_ping_xbar -e <enclosure> -s <slot> [-x <xbar_no>] -p <port> [ -l
   <length>] [ -n <num_pkts>] [-b <board_no> ]*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <ctype.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_dflt_error.h"
#include "lf_xbar32.h"
#include "libmyri.h"
#include "queue.h" 

#define MAX_ROUTE_LEN 32

#define DEFAULT_PKT_LEN  64
#define MAX_PKT_LEN 1016  /* more than this causes a segfault... why? */

#define DEFAULT_NUM_PACKETS  100
#define PING_PACKET_TIMEOUT 100

#define PING_PKT_TYPE 0xabcdef12

struct raw_pkt {
  uint32_t pkt_type;
  struct lf_tagged_xbar_insert x32_stuff;
  uint32_t seqno;
};

struct xbar_mark {
  int visited;
  int take_port;
};
  
int verbose=0;
int debug=0;
int skip_fabric=0;
int retrieve_xb32info=0;

void 
usage(
  char *progname)
{
  printf("Usage: \n"
	 "  %s -e <enclosure> -s <slot> [-x <xbar_no>] -p <port>\n"
	 "         [-l <length>] [-n <num_pkts>] [-b <board_no> ]\n"
	 "         [-i interface] [-v] [-d] [-r <reps>] [-t] \n\n"
	 " or:\n"
	 "  %s [-l <length>] [-n <num_pkts>] [-b <board_no>] \n"
	 "         [-i interface] [-v] [-t] [-I] route_bytes\n\n"
	 "    -e enclosure name\n"
	 "    -s slot number (see Note)\n"
	 "    -x xbar number if the slot has more than 1\n"
	 "    -p port number\n"
	 "    -l length (%d)\n"
	 "    -n number of packets (%d)\n"
	 "    -b number of the board to ping with (0)\n"
	 "    -i which port on the board to ping with (0)\n"
	 "    -v verbose mode\n"
	 "    -d debug mode (use with verbose for more info)\n"
	 "    -r repetitions ( zero is forever )\n"
	 "    -t traceroute mode\n"
	 "    route_bytes - this form of the command sends packets to\n"
	 "       the route specified. Doesn't use the database\n" 
	 "    -S Skip database lookup.  Don't look at the fma database\n"
	 "       and just send data through the route specified.\n"
	 "       By default this won't return xbar32 information\n"
	 "       since there is no way to know if it should be an xbar32.\n"
	 "    -I Retrieve xbar32 information.  This is only useful if\n"
	 "       route bytes are specified AND you are skipping the \n"
	 "       the database lookup AND you are sure the target\n"
         "       is an xbar32.\n\n"
	 "  Note:  xbar16 based switches number the monitoring line card\n"
	 "   as slot 0, while the xbar32 based switches number the \n" 
	 "   the monitoring line card as slot M. For the purposes of \n"
	 "   specifying the slot number for this tool when using it with\n"
	 "   xbar16 switches, slot '0' is the first slot capable of \n"
	 "   holding a myrinet switch line card and not the monitoring \n"
	 "   line card.\n",
	 progname, progname, DEFAULT_PKT_LEN, DEFAULT_NUM_PACKETS);
  exit(1);
}  

static struct lf_nic *
find_this_nic(
  struct lf_fabric *fp, 
  int Myri)
{
  struct myri_nic_info info;
  struct lf_nic *np;
  union lf_node *node;
  struct lf_xbar *xb1;

  int rc;
  int i;

  /* get information including MAC address about this NIC */
  rc = myri_get_nic_info(Myri, &info);
  if (rc != 0) return NULL;

  /* find matching NIC in fabric */
  np = lf_find_nic_by_mac(fp, info.mac_addr);
  if (np == NULL) return NULL;

  if (verbose) {
    struct lf_linecard *lc;
    struct lf_enclosure *ec;
    
    printf("---\n");
    printf("host board %d has mac address " LF_MAC_FORMAT "\n", 
	   np->host_nic_id, LF_MAC_ARGS(np->mac_addr));

    for (i=0; i<np->num_ports; i++) {
      /* print where the host is. for debugging purposes */
      node = np->topo_ports[i];
      if (node == NULL) {
	printf("Error traversing database.\n");
	exit(1);
      }
      
      if (node->ln_type != LF_NODE_XBAR) {
	printf("Error traversing database.\n");
	exit(1);
      }
      
      xb1 = LF_XBAR(node);
      lc = xb1->linecard;
      ec = lc->enclosure;
      printf("Port %d is connected to enclosure %s, slot %d, port %d\n",
	     i, ec->name, lc->slot, np->topo_rports[i]);
    }
    printf("---\n");    
  }
  return np;
}

struct lf_xbar *
find_this_xbar(
  struct lf_fabric *fp, 
  char *enc_name, 
  int slot_num,
  int xbar_num)
{
  struct lf_enclosure *ep;
  struct lf_linecard *slot;
  struct lf_xbar *xbar;

  ep = lf_find_enclosure_by_name(fp, enc_name);
  if (ep == NULL) {
    fprintf(stderr, "Couldn't find the target enclosure in database.\n");
    exit(1);
  }

  /* go the the enclosure's slot*/
  if (slot_num >= ep->lc_slotbase
      && slot_num < ep->lc_slotbase + ep->num_lc_slots) {
    slot_num -= ep->lc_slotbase;
  } else if (slot_num >= ep->bp_slotbase
	     && slot_num < ep->bp_slotbase + ep->num_bp_slots) {
    slot_num = slot_num - ep->bp_slotbase + ep->num_lc_slots;
  } else {
    fprintf(stderr, "According to the database, slot number was too high"
	    "for this enclosure.\n");
    exit(1);
  }

  if (verbose) {
    printf("enclosure %s has %d slots, slot index=%d\n", ep->name,
	   ep->num_lc_slots, slot_num);
  }

  slot = ep->slots[slot_num];
  if (slot == NULL) {
    printf("Slot %d looks like it doesn't exist in the database.\n", 
	   slot_num);
    exit(1);
  }
  if (verbose) {
    printf("slot %d has serial %s\n", slot->slot, slot->serial_no);
  }

  /* now get the xbar */
  if (slot->num_xbars > 1 && xbar_num == -1) {
    fprintf(stderr, "The target linecard has multiple xbars\n"
	    "Please specify one.\n");
    exit(1);
  }
  else if (xbar_num == -1) {
    xbar_num = 0;
  }

  if (xbar_num >= slot->num_xbars || xbar_num < 0) {
      fprintf(stderr, "xbar %d doesn't exist in the database!\n", xbar_num);
      exit(1);
  }

  xbar = LF_XBAR(slot->xbars[xbar_num]);
  if (!xbar) {
    fprintf(stderr, "Error trying to find the xbar in the database!\n");
  }


  if (verbose) {
    printf("xbar %d, num_ports: %d, xbar_id = %d\n", 
	   xbar->xbar_no, xbar->num_ports, xbar->xbar_id);
    printf("---\n");
  }
  return xbar;
}

void
alloc_marks(
  struct lf_fabric *fp)
{
  int i;
  struct lf_xbar *xb;
  for (i=0; i < fp->num_xbars; i++) {
    xb = fp->xbars[i];
    xb->user.v = (void *) calloc(1, sizeof(struct xbar_mark));
    if (xb->user.v == NULL) {
      perror("calloc failed");
    }
    ((struct xbar_mark *)xb->user.v)->take_port = -2;
  }
}

void
free_marks(
  struct lf_fabric *fp)
{
  int i;
  struct lf_xbar *xb;

  for (i=0; i < fp->num_xbars; i++) {
    xb = fp->xbars[i];
    if (xb->user.v) {
      free(xb->user.v);
    }
    xb->user.v = NULL;
  }
}

/* simply mark all the other xbars with the port to take to get to the
   target xbar */

void 
mark_xbars(
  struct lf_xbar *xb) 
{
  int i;
  struct lf_xbar *xb1;
  struct lf_xbar *xb2;
  struct xbar_mark *m1, *m2;
  int rport;

  struct xbar_list {
    struct lf_xbar *xb;
    SLIST_ENTRY(xbar_list) list;
  } *x1, *x2;
  
  SLIST_HEAD(swork,xbar_list) work, newwork;
  
  SLIST_INIT(&work);
  SLIST_INIT(&newwork);
  
  /* fill the worklist with the xbar supplied */
  LF_CALLOC(x1, struct xbar_list, 1);
  x1->xb = xb;
  SLIST_INSERT_HEAD(&work, x1, list);
  
  /* now loop over all the xbars tagging them breadth first */
  while (!SLIST_EMPTY(&work)) {
    while (!SLIST_EMPTY(&work)) {
      x1 = SLIST_FIRST(&work);
      xb1 = x1->xb;
      m1 = (struct xbar_mark *) xb1->user.v;

      /* look for new xbars at the next level */
      for (i=0; i < xb1->num_ports; i++) {
	xb2 = LF_XBAR(xb1->topo_ports[i]);
	rport = xb1->topo_rports[i];
	/* skip everything that is not an xbar */
	if (xb2 == NULL || xb2->ln_type != LF_NODE_XBAR) {
	  continue;
	}
	m2 = (struct xbar_mark *)xb2->user.v;
	/* also skip xbars that we have already visited*/
	if (m2->visited == 1) {
	  continue;
	}

	/* ok this is new, mark it as visited, and add the port to take*/
	m2->visited = 1;
	m2->take_port = rport;

	/*printf("xbar %p has take_port %d\n", xb2, rport);*/
	/* finally add the new xbar to the new worklist */
	LF_CALLOC(x2, struct xbar_list, 1);
	x2->xb = xb2;
	SLIST_INSERT_HEAD(&newwork, x2, list);
      }

      /* remove the xbar we've processed */
      SLIST_REMOVE_HEAD(&work, list);
      free(x1);
    }
    /* set work to the next level of xbars to work on. */
    SLIST_FIRST(&work) = SLIST_FIRST(&newwork);
    SLIST_INIT(&newwork);
  }

  return;

 except:
  exit(1);

}

void
find_route( 
  unsigned char *route, 
  int *route_len, 
  struct lf_fabric *fp, 
  struct lf_nic *nic, 
  struct lf_xbar *xbar, 
  int ifc,
  int xbar_port_num )
{
  struct lf_xbar *xb1, *nic_xb;
  union lf_node *n;
  struct lf_nic *tnic;
  struct xbar_mark *m;
  int i;
  int rport;
  int nic_rport;

  /* check if host is connected to an xbar */
  nic_xb = LF_XBAR(nic->topo_ports[ifc]);
  nic_rport = nic->topo_rports[ifc];

  if ( nic_xb == NULL || nic_xb->ln_type != LF_NODE_XBAR ) {
    fprintf(stderr, "According to the database, the host is not "
	    "connected to an xbar.\n");
    exit (1);
  }

  /* now do a few checks on the target xbar */
  /* first that the port exists */

  if (xbar_port_num > xbar->num_ports) {
    fprintf(stderr, "According to the database, this xbar does not" 
	    "have port %d\n", xbar_port_num);
    exit(1);
  }

  /* check if target xbar's port is connected to anything */
  n = xbar->topo_ports[xbar_port_num];
  rport = xbar->topo_rports[xbar_port_num];
  if ( n == NULL ) {
    fprintf(stderr, "According to the database, the target port looks "
	    "like it is disconnected.\n");
    exit(1);
  }

  /* check the trivial case where the port we want to ping is directly 
     connect to a host, hopefully us */
  if (n->ln_type == LF_NODE_NIC) {
    /* check if this is the host */
    tnic = LF_NIC(n);
    if (tnic == nic) {
      *route_len = 0;
      return;
    }
    else {
      fprintf(stderr, "According to the database, the target port is"
	      "connected to another host.\n");
      exit(1);
    }
  }

  /* check case of the host is connected to the same xbar we want to ping, 
     but on one of its other ports */
  /* TODO ask reese if this is acceptable, since it is a very bad route 
     that goes over the same link too many times */
  if (nic_xb == xbar) {
    route[0] = LF_DELTA_TO_ROUTE( xbar_port_num - nic_rport);
    route[1] = 0x80;
    *route_len = 2;
    return;
  }

  /* All the sanity checks and trivial stuff has now been checked so
     lets see if we can find a route */
  /* ok first we are going to mark every xbar with which port to take
     to get back to the starting xbar */
  
  xb1 = LF_XBAR(n);
  alloc_marks(fp);
  /* mark the original xbar by hand */
  m = (struct xbar_mark *)xbar->user.v;
  m->visited = 1;
  m->take_port = -1; /* signal that this is the target */

  /* mark the next xbar by hand since we got here by hand */
  m = (struct xbar_mark *)xb1->user.v;
  m->visited = 1;
  m->take_port = rport;

  mark_xbars(xb1);
  
  /* now we follow the bread crumbs from the host back to our target
     xbar building up the route as we go along */
  xb1 = nic_xb;
  rport = nic_rport;
  i = 0;
  *route_len = 0;

  while (1) {
    if (i > MAX_ROUTE_LEN) {
      fprintf(stderr, "Something went wrong with route selection.\n"
	      "The route selected is longer than supported\n");
      exit(1);
    }
    m = (struct xbar_mark *)xb1->user.v;  
    /*printf("xb %p tell me to take port %d\n",xb1, m->take_port); */
    if (m->take_port == -1) {
      break;
    }
    else if (m->take_port == -2) {
      fprintf(stderr, "It looks like there might not be a route between"
	      "the target xbar and the host\n");
      exit(1);
    }
    route[i] = LF_DELTA_TO_ROUTE(m->take_port - rport);
    ++(*route_len);
    rport = xb1->topo_rports[m->take_port];
    xb1 = LF_XBAR(xb1->topo_ports[m->take_port]);
    i++;
  } 

  free_marks(fp);

}

static inline void
print_route( 
  char *message,
  unsigned char *route,
  int len)
{
  int i;

  if (message) printf("%s ", message);
  if (verbose) printf("(len %d) : ", len);
  for (i=0; i<len; i++) {
    printf("%02x ", (unsigned char) route[i]);
  }
  printf("\n");
}
 
/* walk the route to figure out where we are heading */
int
walk_route(
  struct lf_fabric *fabric,
  struct lf_nic *nic,
  int ifc,
  unsigned char *route,
  int len,
  uint32_t *serial)
{
  struct lf_xbar *xb;
  struct lf_xbar *xb_next;
  int rport;
  int rport_next;
  int i;
  int next_port;

  if (verbose && debug) {
    print_route("Walking Route", route, len);
  }

  xb = LF_XBAR(nic->topo_ports[ifc]);
  rport = nic->topo_rports[ifc];

  if (xb == NULL) {
    fprintf(stderr, "Your host nic is disconnected in the database. \n");
    exit(1);
  }
  if (xb->ln_type != LF_NODE_XBAR) {
    fprintf(stderr, "Ran into something that wasn't an xbar!\n");
    exit(1);
  }

  for (i=0; i<len; i++) {
    next_port = rport + LF_ROUTE_TO_DELTA(route[i]);
    if (next_port < 0 || next_port >= xb->num_ports) {
      fprintf(stderr, "Error with the route supplied."
	      " Please check the route.\n");
      exit(1);
    }
    xb_next = LF_XBAR(xb->topo_ports[next_port]);
    rport_next = xb->topo_rports[next_port];

    if (xb_next == NULL) {
      fprintf(stderr, 
	      "A link along the route is disconnected in the database.\n"
	      "Route[%d] = %02x (%d) is the byte in question.\n",
	      i, (unsigned char) route[i], LF_ROUTE_TO_DELTA(route[i]));
      exit(1);
    }
    if (xb_next->ln_type != LF_NODE_XBAR) {
      fprintf(stderr, "Ran into something that wasn't an xbar!\n");
      exit(1);
    }
    xb = xb_next;
    rport = rport_next;
  }
 
  if (xb->num_ports == 32) {
    if (xb->linecard) {
      *serial = xb->xbar_id;
      if (verbose) {
	printf("Target xbar should be an xbar32. xbar_id should be = %d\n"
	       "---\n", *serial);
      }
      return 1;
    }
    else {
      fprintf(stderr, "Error finding linecard associated with target xbar!\n");
      exit(1);
    }
  }
  else {
    *serial=-1;
    if (verbose) {
      printf("Target xbar is not an xbar32\n---\n");
    }
    return 0;
  }
}

void
ping_route( 
  unsigned char *route, 
  int route_len, 
  int len, 
  int num_packets, 
  int myri, 
  int ifc,
  struct lf_tagged_xbar_insert *xbdata)
{
  void *vbuf;
  struct raw_pkt *message;
  struct timeval start, end;
  uint64_t elapsed_us;
  float bw;

  int rx_packets = 0;
  int status;
  int i;

  if (verbose && debug) {
    print_route("Ping Route", route, route_len);
  }

  LF_CALLOC(vbuf, char, len);
  message = vbuf;

  message->pkt_type = PING_PKT_TYPE;

  /* zero out the bytes to be replaced by the x32 HW so that CRC32 comes out
     right... do this even everytime just in case.  Thanks reese for
     documenting this in your linktest program... otherwise i would have
     spent hours trying to figure out why it wasn't working */
  memset(&message->x32_stuff, 0, sizeof(message->x32_stuff));

  for (i=0; i<num_packets; i++) {
    message->seqno = i;
    status = myri_raw_send(myri, ifc, route, route_len, message, len, NULL);
    if (status) {
      perror("myri_raw_send");
      exit(1);
    }
  }

  status = gettimeofday(&start, NULL);
  if (status) {
    perror("gettimeofday");
    exit(1);
  }

  while (1) {
    /* get events */
    struct myri_event *mep;
    status = myri_next_event(myri, &mep, PING_PACKET_TIMEOUT);
    if (status) {
      perror("myri_next_event");
      exit(1);
    }

    /* no event means a timeout */
    if (mep->type == MYRI_EVENT_NO_EVENT) {
      /* no need to release non-events */
      break;
    }
    
    /* process receives */
    if (mep->type == MYRI_EVENT_RAW_RECV_COMPLETE) {
      struct raw_pkt *pp;
      pp = mep->d.raw_recv.rxbuf;

      /* process this packet if it's ours */
      if (pp->pkt_type == PING_PKT_TYPE) { 
	++rx_packets;
	if (pp->seqno == num_packets - 1) {
	  if (xbdata) {
	    memcpy(xbdata, &pp->x32_stuff, sizeof(struct lf_tagged_xbar_insert));
	  }
	  myri_release_event(mep);		/* release this event */
	  break;
	}
      }
    }
    
    myri_release_event(mep);
  }

  status = gettimeofday(&end, NULL);
  if (status) {
    perror("gettimeofday\n");
    exit(1);
  }

  /* computer bandwidth */
  elapsed_us = (end.tv_sec - start.tv_sec) * 1000000
    + end.tv_usec - start.tv_usec;

  if (verbose) {
    printf("Time %d us\n", (int)elapsed_us);
  }
  if (elapsed_us > 0) {
    bw = ((double )len * (double)rx_packets) / (double)elapsed_us;
    if (!verbose) printf("%5.1f MB/s ", bw);
    else printf("Bandwidth is %5.1f MB/s  ", bw);
  }
  else {
    if (!verbose) printf("NAN  MB/s");
    else printf("Elapsed time is zero! Not reporting BW");
  }

  if (!verbose) {
    printf ("%5d/%-5d   %3.0f%%    ",
	    rx_packets, num_packets, 
	    ((double) 1 - ((double)rx_packets/num_packets)) * 100);
    print_route(NULL, route, route_len/2);
  }
  else {
      printf ("Received %4d/%4d (%3.0f%% loss)\n",
	    rx_packets, num_packets, 
	    ((double) 1 - ((double)rx_packets/num_packets)) * 100);
  }  
  return;
    
  free(message);
  
 except:
  exit(1);
}

void
return_route(
  unsigned char *route, 
  int *route_len, 
  int want_xb32_data)
{
  int i;

  if (want_xb32_data || retrieve_xb32info) {
    route[*route_len] = 0xA0;
  }
  else {
    route[*route_len] = 0x80;
  }
  for (i=0; i<*route_len; i++) {
    route[*route_len + i + 1] = 
      LF_DELTA_TO_ROUTE( -LF_ROUTE_TO_DELTA(route[*route_len - i - 1]));
  }
  *route_len = *route_len * 2 + 1;

}

/* stolen shamelessly from  fm_walkroute.c */
int
ascii_route_to_delta(
  char *route)
{
  int r;
  char *cp;
  int nondec;

  nondec=0;
  cp = route;
  if (*cp == '-') ++cp;
  
  while (*cp != '\0') {
    if (!isdigit(*cp)) {
	nondec=1;
	break;
    }
    ++cp;
  }

  if (*route == '-' && nondec) {
    fprintf(stderr, "Bad route byte: %s\n", route);
    exit(1);
  }

  if (!nondec) {
    r = atoi(route);
    if (r < 80) return r;
  }

  sscanf(route, "%x", &r);
  r &= 0x3f;
  if (r & 0x20) {
    r |= (~0 & ~0x3f);		/* sign extend */
  }

  return r;
}

void
check_xid(
  uint32_t target_id,
  struct lf_tagged_xbar_insert *xbdata)
{
  uint32_t xid;

  xid = lf_swap_l(ntohl(xbdata->id));
  
  if (target_id != -1 && xid != target_id) {
    printf("WARNING: xbar ids didn't match!\n");
    printf("WARNING: xbar returned %d, expected %d\n",
	   xid, target_id);
  }
}

void 
print_xbinfo(
  struct lf_tagged_xbar_insert *xbdata)
{
  printf("---\n");
  printf("Info returned by xbar:\n");
  printf("id: %d\n", 
	 lf_swap_l(ntohl(xbdata->id)));
  printf("port: %d\n", xbdata->absolute_port);
  if (debug) {
    printf("quadrant disable: %x\n", xbdata->quadrant_disabled);
    printf("status_mask: 0x%08x\n", 
	   lf_swap_l(ntohl(xbdata->status_mask)));
  }
  printf("---\n");
}

int 
main(
  int argc, 
  char *argv[])
{
  int c;
  char *enc_name = NULL;
  int slot_num = -1;
  int xbar_num = -1;
  int port_num = -1;
  unsigned int length = DEFAULT_PKT_LEN;
  unsigned int num_packets = DEFAULT_NUM_PACKETS;
  unsigned int reps = 1;
  int board_num = 0;
  unsigned char route[MAX_ROUTE_LEN];
  int route_len = -1;
  int ifc = 0;
  int traceroute = 0;
  uint32_t target_id;
  int is_xb32 = 0;
  unsigned char tmp_route[MAX_ROUTE_LEN * 2 + 1];
  int t_len;
  char *fms_run;
  char *db_name;

  struct lf_fabric *fabric = NULL;
  static struct lf_nic *nic = NULL;
  struct lf_xbar *xbar = NULL;

  struct lf_tagged_xbar_insert xbdata;

  int myri = -1;
  
  int i, j;

  lf_init();

  /* clear the route */
  bzero(route, MAX_ROUTE_LEN);
  db_name = NULL;
  fms_run = NULL;

  while ((c = getopt(argc, argv, "H:R:N:e:s:x:p:l:n:b:i:r:vdtIS")) != EOF) {
    switch (c) {

  case 'H':
    fprintf(stderr,
	"Please note: -H has been replaced with -R and is deprecated\n");
    /* FALLSTHROUGH */

  case 'R':
    fms_run = optarg;
    break;

  case 'N':
    db_name = optarg;
    break;

    case 'e':
      enc_name = optarg;
      break;
    case 's':
      slot_num = atoi(optarg);
      break;
    case 'x':
      xbar_num = atoi(optarg);
      /* support boards up to 16 xbars... even though only
	 4 xbars is the current max.  But the EEs have reserved
	 up to 16 xbar ids per card so there could be future expansion */
      if (xbar_num < 0 || xbar_num > 15) {
	usage(argv[0]);
      }
      break;
    case 'p':
      port_num = atoi(optarg);
      break;
    case 'l':
      length = atoi(optarg);
      if (length > MAX_PKT_LEN) {
	fprintf(stderr, "Maximum length is %d\n",
		MAX_PKT_LEN);
	exit(1);
      }
      break;
    case 'n':
      num_packets = atoi(optarg);
      break;
    case 'b':
      board_num = atoi(optarg);
      break;
    case 'r':
      reps = abs(atoi(optarg));
      break;
    case 'i':
      ifc = atoi(optarg);
      break;
    case 'v':
      verbose = 1;
      break;
    case 't':
      traceroute = 1;
      break;
    case 'S':
      skip_fabric = 1;
      break;
    case 'I':
      retrieve_xb32info = 1;
      break;
    case 'd':
      debug++;
      break;
    default:
      usage(argv[0]);
    }
  }

  /* set dirs - doesn't care if they are NULL */
  lf_simple_set_fabric_dirs(fms_run, db_name);
  
  if (length < sizeof(struct raw_pkt)) {
    fprintf(stderr, "Minimum packet length is %d\n",
	    (int) sizeof(struct raw_pkt));
    exit(1);
  }

  /* first open the myrinet card */
  myri = myri_open(board_num);
  if (myri == -1) {
    perror("myri_open");
    exit(1);
  }

  /* next load the fabric if needed */
  if (!skip_fabric) {
    /* now load the fabric */
    fabric = lf_simple_load_fabric();
    if (fabric == NULL) {
      fprintf(stderr, "Error loading fabric\n");
      exit(1);
    }
    /* find ourselves in the fabric */
    nic = find_this_nic(fabric, myri);
    if (nic == NULL) {
      fprintf(stderr, "Error finding ourselves.\n");
      exit(1);
    }
  }

  /* were we called with a route? */
  /* note specifying no route should also be legal! */
  if (enc_name == NULL && slot_num < 0 &&
      port_num < 0 && xbar_num < 0) {
    /* we were probably called with a route */
    route_len = 0;
    while (optind < argc) {
      route[route_len++] = 
	LF_DELTA_TO_ROUTE(ascii_route_to_delta(argv[optind++]));
      if (route_len >= MAX_ROUTE_LEN) {
	fprintf(stderr, "Error. Route is longer than %d "
		"the currently supported maximum route len.\n", MAX_ROUTE_LEN);
      }
    }
  }
  /* nope we don't have a route */
  else {
    /* leave the decision of the xbar number until we know
       if the linecard has multiple xbars */
    if (enc_name == NULL || slot_num < 0 || port_num < 0 || skip_fabric) {
      usage(argv[0]);
    }

    if (verbose) {
      printf("Enc: %s, slot: %d, xbar: %d, port: %d\n", enc_name, slot_num, 
	     xbar_num, port_num);
      if (debug) {
	printf("Length: %d, packets: %d, board: %d, ifc:  %d\n", 
	       length, num_packets, board_num, ifc);
      }
    }

    if ( lf_build_xbar_array(fabric)) {
      fprintf(stderr, "Error building xbar array\n");
    }

    /* find our target in the fabric */
    xbar = find_this_xbar(fabric, enc_name, slot_num, xbar_num);
    if (xbar == NULL) {
      fprintf(stderr, "Error finding target xbar\n");
      exit(1);
    }
    /* figure out how to get from us to the target */
    route_len = -1;
    find_route(route, &route_len, fabric, nic, xbar, ifc, port_num);
    if (route_len == -1) {
      fprintf(stderr, "Error calculating a route\n");
      exit(1);
    }
  }  

  if (!verbose) printf(" Bandwidth  Recv/Sent    Loss    Route\n");

  /* repetitions loop*/
  for (j=0; j<reps || reps == 0; j++) {

    if (traceroute) i = 0;
    else i = route_len;

    /* traceroute loop*/
    for (; i<=route_len; i++) {
      t_len = i;
      memcpy(tmp_route, route, t_len);

      if (verbose) print_route("Route", tmp_route, t_len);

      /* check where the route goes */
      if (!skip_fabric) {
	is_xb32 = walk_route(fabric, nic, ifc, tmp_route, t_len, &target_id);
      }

      /* modify our route so that the packets come back */
      return_route(tmp_route, &t_len, is_xb32);

      /* send stuff to the target and calculate BW and packet loss */
      ping_route(tmp_route, t_len, length, num_packets, myri, ifc, &xbdata);

      /* check the info returned */
      if (is_xb32) {
	check_xid(target_id, &xbdata);
      }
      if (verbose && (is_xb32 || retrieve_xb32info)) {
	print_xbinfo(&xbdata);
      }
    }
  }

  /* closing things properly */

  myri_close(myri);  /*Currently broken? */

  if (!skip_fabric) {
    lf_free_fabric(fabric);
  }

  return 0;
}

    
